<?php

namespace App\Http\Controllers;

use App\Models\Candidate;
use App\Models\Survey;
use App\Models\Vote;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;

class VoteController extends Controller
{
    public function store(Request $request)
    {
        // RF04: Validación del formato del RUT y candidato
        $request->validate([
            'rut' => [
                'required',
                'string',
                function ($attribute, $value, $fail) {
                    if (!$this->validateRut($value)) {
                        $fail('El formato del RUT no es válido.');
                    }
                }
            ],
            'candidate_name' => 'required|string|min:2|max:255',
            'reason' => 'required|string|min:10|max:1000',
            'survey_id' => 'required|exists:surveys,id'
        ], [
            'rut.required' => 'Debe ingresar su RUT.',
            'candidate_name.required' => 'Debe ingresar el nombre de su candidato.',
            'reason.required' => 'Debe incluir una pequeña reseña del por qué selecciona a esta persona.',
            'reason.min' => 'La reseña debe tener al menos :min caracteres.',
            'reason.max' => 'La reseña no debe exceder los :max caracteres.'
        ]);

        $rutHash = hash('sha256', strtoupper(preg_replace('/[^0-9K]/i', '', $request->rut)));
        $surveyId = $request->survey_id;
        $candidateName = trim($request->candidate_name);

        // RF05: Validar que el RUT no haya votado en la misma encuesta (Concurrencia segura)
        try {
            DB::transaction(function () use ($surveyId, $request, $rutHash, $candidateName) {

                $hasVoted = Vote::where('survey_id', $surveyId)
                    ->where('rut_hash', $rutHash)
                    ->exists();

                if ($hasVoted) {
                    throw new \Exception('El RUT ingresado ya ha votado en esta encuesta. Solo se permite un voto por persona.');
                }

                // Buscar o crear candidato (ignorando mayúsculas/minúsculas)
                $candidate = Candidate::where('survey_id', $surveyId)
                    ->whereRaw('LOWER(name) = ?', [strtolower($candidateName)])
                    ->first();

                if (!$candidate) {
                    $candidate = Candidate::create([
                        'survey_id' => $surveyId,
                        'name' => $candidateName,
                    ]);
                }

                Vote::create([
                    'survey_id' => $surveyId,
                    'candidate_id' => $candidate->id,
                    'rut_hash' => $rutHash,
                    'reason' => $request->reason
                ]);
            });

            // RF08: Confirmación de voto
            return redirect()->route('survey.show')->with('success', '¡Su voto ha sido registrado correctamente!');
        } catch (\Exception $e) {
            return back()->withErrors(['rut' => $e->getMessage()])->withInput();
        }
    }

    // Reglas de Módulo 11 estándar Chileno
    private function validateRut($rut)
    {
        $rut = preg_replace('/[^0-9Kk]/i', '', $rut);
        if (strlen($rut) < 8)
            return false;

        $dv = substr($rut, -1);
        $number = substr($rut, 0, -1);

        $i = 2;
        $sum = 0;
        foreach (array_reverse(str_split($number)) as $v) {
            if ($i == 8)
                $i = 2;
            $sum += $v * $i;
            ++$i;
        }

        $calcDv = 11 - ($sum % 11);
        if ($calcDv == 11)
            $calcDv = 0;
        if ($calcDv == 10)
            $calcDv = 'K';

        return strtoupper($dv) == strtoupper($calcDv);
    }
}
